/**************************************************************************
 *
 * Copyright 2010, 2011 BMW Car IT GmbH 
 * Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include "egl_helper.h"
#include "ilm_control.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>

#include <stdint.h>
#include <signal.h>
#include <sys/stat.h>
#include <linux/fb.h>

#include <iostream>

#include "WaylandServerinfoClientProtocol.h"

using namespace std;

/* #define USE_PIXMAP */

static void serverinfoListener(void *data, struct serverinfo *pServerinfo, uint32_t client_handle)
{
    pServerinfo = pServerinfo; // TODO:to avoid warning
    WLContextStruct* p_wlCtx = (WLContextStruct*)data;
    p_wlCtx->connect_id = client_handle;
    fprintf( stderr, "notified wayland connection : id=%d\n", p_wlCtx->connect_id);
}

struct serverinfo_listener serverinfo_listener_list = {
    serverinfoListener
};

static void registry_handle_global(void* data, struct wl_registry* registry, uint32_t name, const char* interface, uint32_t version)
{
    version = version; // TODO: to avoid warning
    WLContextStruct* p_wlCtx = (WLContextStruct*)data;
    int ans_strcmp = 0;

    do
    {
        ans_strcmp = strcmp(interface, "wl_compositor");
        if (0 == ans_strcmp)
        {
            p_wlCtx->wlCompositor = (wl_compositor*)wl_registry_bind(registry, name, &wl_compositor_interface, 1);
            break;
        }

        ans_strcmp = strcmp(interface, "serverinfo");
        if (0 == ans_strcmp)
        {
            p_wlCtx->wlExtServerinfo = (struct serverinfo*)wl_registry_bind(registry, name, &serverinfo_interface, 1);
            serverinfo_add_listener(p_wlCtx->wlExtServerinfo, &serverinfo_listener_list, data);
            serverinfo_get_connection_id(p_wlCtx->wlExtServerinfo);
        }
    } while(0);
}

static const struct wl_registry_listener registry_listener = {
    registry_handle_global,
    NULL
};

WLContextStruct *createWLContext(t_ilm_int width, t_ilm_int height)
{
    WLContextStruct *context = (WLContextStruct *)calloc (1, sizeof(WLContextStruct));
    if (context == NULL)
    {
        cerr << "Failed to allocate memory for wayland context struct" << endl;
        return NULL;
    }

    context->width = width;
    context->height = height;
    context->wlDisplay = wl_display_connect(NULL);
    if (NULL == context->wlDisplay)
    {
        cerr << "wl_display_connect() failed" << endl;
        destroyWLContext(context);
        return NULL;
    }

    context->wlRegistry = wl_display_get_registry(context->wlDisplay);
    wl_registry_add_listener(context->wlRegistry, &registry_listener, context);
    wl_display_dispatch(context->wlDisplay);
    wl_display_roundtrip(context->wlDisplay);

    context->wlSurface = wl_compositor_create_surface(context->wlCompositor);
    if (NULL == context->wlSurface)
    {
        cerr << "wl_compositor_create_surface() failed" << endl;
        destroyWLContext(context);
        return NULL;
    }

    context->wlNativeWindow = wl_egl_window_create(context->wlSurface, width, height);
    if (NULL == context->wlNativeWindow)
    {
        cerr << "wl_egl_window_create() failed" << endl;
        destroyWLContext(context);
        return NULL;
    }

    return context;
}

EGLint contextAttribs[] = { EGL_CONTEXT_CLIENT_VERSION, 2, EGL_NONE };

EglContextStruct *createEGLContext(t_ilm_int width, t_ilm_int height,
                                   t_ilm_layer layerid,
                                   t_ilm_surface surfaceid,
                                   WLContextStruct *wl_context)
{
    EglContextStruct *context = (EglContextStruct *)calloc(1, sizeof(EglContextStruct));

    context->eglDisplay = NULL;
    context->eglSurface = NULL;
    context->eglContext = NULL;
    context->surfaceid = surfaceid;
    ilmErrorTypes error = ILM_FAILED;
    uint32_t native_ilm_handle = 0;
    EGLint eglstatus = EGL_SUCCESS;

    eglstatus = eglGetError(); // Clear error

    context->eglDisplay = eglGetDisplay(wl_context->wlDisplay);
    if (!context->eglDisplay)
    {
        cerr << "eglGetDisplay() failed" << endl;
        destroyEglContext(context);
        return NULL;
    }

    EGLint iMajorVersion, iMinorVersion;
    if (!eglInitialize(context->eglDisplay, &iMajorVersion,
            &iMinorVersion))
    {
        eglstatus = eglGetError();
        cerr << "eglInitialize() failed, code 0x" << hex << eglstatus<< endl;
        destroyEglContext(context);
        return NULL;
    }

    eglBindAPI(EGL_OPENGL_ES_API);
    eglstatus = eglGetError();
    if (eglstatus != EGL_SUCCESS)
    {
        cerr << "eglBindAPI() failed" << endl;
        printf("Error: eglBindAPI() failed.\n");
        if (eglstatus == EGL_BAD_PARAMETER)
            cerr << "EGL_OPENGL_ES_API is not supported "
                    "by the EGL implementation" << endl;

    }
    EGLint pi32ConfigAttribs[] = {
        EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
        EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
        EGL_RED_SIZE,   8,
        EGL_GREEN_SIZE, 8,
        EGL_BLUE_SIZE,  8,
        EGL_ALPHA_SIZE, 8,
        EGL_NONE
    };
    int iConfigs;

    if (!eglChooseConfig(context->eglDisplay, pi32ConfigAttribs, &context->eglConfig, 1, &iConfigs) || (iConfigs != 1))
    {
        eglstatus = eglGetError();
        cerr << "eglChooseConfig() failed, code 0x"
             << hex << eglstatus << endl;
        destroyEglContext(context);
        return NULL;
    }

    context->eglSurface = eglCreateWindowSurface(
            context->eglDisplay, context->eglConfig,
            wl_context->wlNativeWindow, NULL);
    eglstatus = eglGetError();

    if (eglstatus != EGL_SUCCESS)
    {
        cerr << "eglCreateWindowSurface() failed, code 0x"
             << hex << eglstatus << endl;
        destroyEglContext(context);
        return NULL;
    }

    context->eglContext = eglCreateContext(
            context->eglDisplay, context->eglConfig, NULL, contextAttribs);

    eglstatus = eglGetError();
    if (eglstatus != EGL_SUCCESS)
    {
        cerr << "eglCreateContext() failed, code 0x"
             << hex << eglstatus << endl;
        destroyEglContext(context);
        return NULL;
    }

    eglMakeCurrent(context->eglDisplay,
            context->eglSurface, context->eglSurface,
            context->eglContext);
    eglSwapInterval(context->eglDisplay, 1);
    eglstatus = eglGetError();
    if (eglstatus != EGL_SUCCESS)
    {
        cerr << "eglMakeCurrent() failed, code 0x" << hex << eglstatus << endl;
        destroyEglContext(context);
        return NULL;
    }

    // TODO: if (error == ILM_FAILED) return ILM_FALSE;

    // TODO: auto generate surface id
    struct wl_proxy* pxy = (struct wl_proxy*)wl_context->wlSurface;
    uint32_t id = wl_proxy_get_id(pxy);
    native_ilm_handle = (wl_context->connect_id << 16) | id;
    error = ilm_surfaceCreate( (t_ilm_nativehandle) native_ilm_handle, width, height,
            ILM_PIXELFORMAT_RGBA_8888, &surfaceid);

    if (error != ILM_SUCCESS)
    {
        cerr << "ilm_surfaceCreate failed: " << ILM_ERROR_STRING(error) << endl;
        destroyEglContext(context);
        return NULL;
    }

    error = ilm_surfaceSetDestinationRectangle(surfaceid, 0, 0, width, height);
    if (error != ILM_SUCCESS)
    {
        cerr << "ilm_surfaceSetDestinationRectangle failed: "
             << ILM_ERROR_STRING(error) << endl;
        destroyEglContext(context);
        return NULL;
    }


    error = ilm_surfaceSetSourceRectangle(surfaceid, 0, 0, width, height);
    if (error != ILM_SUCCESS)
    {
        cerr << "ilm_surfaceSetSourceRectangle failed: "
             << ILM_ERROR_STRING(error) << endl;
        destroyEglContext(context);
        return NULL;
    }

    error = ilm_layerAddSurface(layerid, surfaceid);
    if (error != ILM_SUCCESS)
    {
        cerr << "ilm_layerAddSurface failed: "
             << ILM_ERROR_STRING(error) << endl;
        destroyEglContext(context);
        return NULL;
    }

    error = ilm_surfaceSetVisibility(surfaceid, ILM_TRUE);
    if (error != ILM_SUCCESS)
    {
        cerr << "ilm_surfaceSetVisibility failed: "
             << ILM_ERROR_STRING(error) << endl;
        destroyEglContext(context);
        return NULL;
    }

    error = ilm_surfaceSetOpacity(surfaceid, 1.0f);
    if (error != ILM_SUCCESS)
    {
        cerr << "ilm_surfaceSetOpacity failed: "
             << ILM_ERROR_STRING(error) << endl;
        destroyEglContext(context);
        return NULL;
    }

    error = ilm_commitChanges();
    if (error != ILM_SUCCESS)
    {
        cerr << "ilm_commitChanges failed: "
             << ILM_ERROR_STRING(error) << endl;
        destroyEglContext(context);
        return NULL;
    }

    return context;
}

void destroyEglContext(EglContextStruct *context)
{
    if (context->eglDisplay != NULL)
    {
        eglMakeCurrent(context->eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
        eglTerminate(context->eglDisplay);
    }
    ilm_surfaceRemove(context->surfaceid);
    ilm_commitChanges();
    free(context);
}

void destroyWLContext(WLContextStruct* context)
{
    if (context->wlNativeWindow)
    {
        wl_egl_window_destroy(context->wlNativeWindow);
    }
    if (context->wlSurface)
    {
        wl_surface_destroy(context->wlSurface);
    }
    if (context->wlCompositor)
    {
        wl_compositor_destroy(context->wlCompositor);
    }
    free(context);
}

t_ilm_uint GetTickCount()
{
    struct timeval ts;
    gettimeofday(&ts, 0);
    return (t_ilm_uint) (ts.tv_sec * 1000 + (ts.tv_usec / 1000));
}

void swapBuffers(WLContextStruct *wlContext, EglContextStruct *eglContext)
{
    eglSwapBuffers(eglContext->eglDisplay, eglContext->eglSurface);

}
